#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import math
import random
import re

import inkex
from inkex import Transform
from inkex.elements import Group, PathElement, Rectangle
from inkex.paths import Path, CubicSuperPath
from lxml import etree


# ---------------------------
# Math helpers
# ---------------------------
def dist(a, b):
    return math.hypot(a[0] - b[0], a[1] - b[1])


def same_pt(a, b, eps=1e-4):
    return abs(a[0] - b[0]) < eps and abs(a[1] - b[1]) < eps


def clamp(v, lo, hi):
    return max(lo, min(hi, v))


# ---------------------------
# Color helpers (robust via inkex.Color)
# ---------------------------
def clamp255(v):
    return max(0, min(255, int(round(v))))


def rgb_to_css(rgb):
    return f"rgb({int(rgb[0])},{int(rgb[1])},{int(rgb[2])})"


def parse_css_color_to_rgb(s):
    if not s:
        return None
    s = str(s).strip()
    if s.lower() in ("none", "transparent"):
        return None
    if s == "currentColor":
        return None
    try:
        c = inkex.Color(s)
        r, g, b = c.to_rgb()
        return (int(r), int(g), int(b))
    except Exception:
        m = re.match(r"^#([0-9a-fA-F]{6})$", s)
        if m:
            h = m.group(1)
            return (int(h[0:2], 16), int(h[2:4], 16), int(h[4:6], 16))
    return None


def rgb_to_hsl(r, g, b):
    r /= 255.0
    g /= 255.0
    b /= 255.0
    mx = max(r, g, b)
    mn = min(r, g, b)
    l = (mx + mn) / 2.0

    if mx == mn:
        return (0.0, 0.0, l)

    d = mx - mn
    s = d / (2.0 - mx - mn) if l > 0.5 else d / (mx + mn)

    if mx == r:
        h = (g - b) / d + (6.0 if g < b else 0.0)
    elif mx == g:
        h = (b - r) / d + 2.0
    else:
        h = (r - g) / d + 4.0
    h *= 60.0
    return (h, s, l)


def hue2rgb(p, q, t):
    if t < 0:
        t += 1
    if t > 1:
        t -= 1
    if t < 1 / 6:
        return p + (q - p) * 6 * t
    if t < 1 / 2:
        return q
    if t < 2 / 3:
        return p + (q - p) * (2 / 3 - t) * 6
    return p


def hsl_to_rgb(h, s, l):
    h = (h % 360.0) / 360.0
    if s == 0.0:
        r = g = b = l
    else:
        q = l * (1 + s) if l < 0.5 else l + s - l * s
        p = 2 * l - q
        r = hue2rgb(p, q, h + 1 / 3)
        g = hue2rgb(p, q, h)
        b = hue2rgb(p, q, h - 1 / 3)
    return (clamp255(r * 255), clamp255(g * 255), clamp255(b * 255))


def rotate_hue_rgb(rgb, deg):
    h, s, l = rgb_to_hsl(rgb[0], rgb[1], rgb[2])
    h = (h + deg) % 360.0
    return hsl_to_rgb(h, s, l)


def is_neutral_gray(rgb, tol=2):
    return abs(rgb[0] - rgb[1]) <= tol and abs(rgb[1] - rgb[2]) <= tol


# ---------------------------
# RDP simplification
# ---------------------------
def point_line_dist_sq(P, A, B):
    vx, vy = (B[0] - A[0]), (B[1] - A[1])
    wx, wy = (P[0] - A[0]), (P[1] - A[1])

    c1 = vx * wx + vy * wy
    if c1 <= 0:
        return (P[0] - A[0]) ** 2 + (P[1] - A[1]) ** 2

    c2 = vx * vx + vy * vy
    if c2 <= c1:
        return (P[0] - B[0]) ** 2 + (P[1] - B[1]) ** 2

    t = c1 / c2
    proj = (A[0] + t * vx, A[1] + t * vy)
    return (P[0] - proj[0]) ** 2 + (P[1] - proj[1]) ** 2


def rdp(points, epsilon):
    if not points or len(points) < 3:
        return points

    closed = same_pt(points[0], points[-1])
    work = points[:-1] if closed else points[:]

    keep = [False] * len(work)
    keep[0] = True
    keep[-1] = True
    eps_sq = epsilon * epsilon

    def rec(a, b):
        if b <= a + 1:
            return
        A = work[a]
        B = work[b]
        max_d = -1.0
        idx = -1
        for i in range(a + 1, b):
            d = point_line_dist_sq(work[i], A, B)
            if d > max_d:
                max_d = d
                idx = i
        if max_d > eps_sq and idx != -1:
            keep[idx] = True
            rec(a, idx)
            rec(idx, b)

    rec(0, len(work) - 1)
    out = [work[i] for i in range(len(work)) if keep[i]]
    if closed and out:
        out.append(out[0])
    return out


# ---------------------------
# Sequential filters
# ---------------------------
def enforce_min_distance_sequential(pts, min_dist):
    if not pts:
        return pts
    out = [pts[0]]
    for p in pts[1:]:
        if dist(p, out[-1]) >= min_dist:
            out.append(p)
    return out


def filter_by_critical_radius_sequential(pts, crit_radius):
    if not pts:
        return pts
    out = []
    i = 0
    while i < len(pts):
        ref = pts[i]
        out.append(ref)
        i += 1
        while i < len(pts) and dist(pts[i], ref) < crit_radius:
            i += 1
    return out


# ---------------------------
# Sampling along CubicSuperPath
# ---------------------------
def cubic_bezier_point(P0, P1, P2, P3, t):
    mt = 1 - t
    a = mt * mt * mt
    b = 3 * mt * mt * t
    c = 3 * mt * t * t
    d = t * t * t
    return (
        a * P0[0] + b * P1[0] + c * P2[0] + d * P3[0],
        a * P0[1] + b * P1[1] + c * P2[1] + d * P3[1],
    )


def approx_bezier_len(P0, P1, P2, P3, steps=12):
    prev = P0
    length = 0.0
    for i in range(1, steps + 1):
        t = i / steps
        p = cubic_bezier_point(P0, P1, P2, P3, t)
        length += dist(prev, p)
        prev = p
    return length


def sample_csp(csp, step_px, treat_closed=False):
    out = []
    if not csp:
        return out

    step_px = max(0.2, float(step_px))

    for sub in csp:
        if len(sub) < 2:
            continue

        closed = treat_closed or same_pt(tuple(sub[0][1]), tuple(sub[-1][1]))
        n = len(sub)
        seg_count = n if closed else (n - 1)

        for i in range(seg_count):
            a = sub[i]
            b = sub[(i + 1) % n]

            P0 = tuple(a[1])
            P1 = tuple(a[2])
            P2 = tuple(b[0])
            P3 = tuple(b[1])

            if not out:
                out.append(P0)

            is_curve = not (same_pt(P0, P1) and same_pt(P2, P3))
            if not is_curve:
                out.append(P3)
            else:
                approx_len = approx_bezier_len(P0, P1, P2, P3)
                steps = max(2, int(math.ceil(approx_len / step_px)))
                for s in range(1, steps + 1):
                    t = s / steps
                    out.append(cubic_bezier_point(P0, P1, P2, P3, t))

    dedup = []
    for p in out:
        if not dedup or dist(p, dedup[-1]) > 1e-6:
            dedup.append(p)
    return dedup


# ---------------------------
# SVG utilities
# ---------------------------
def make_polyline_path(pts, closed=False):
    p = Path()
    if not pts:
        return p
    p.append(inkex.paths.Move(pts[0][0], pts[0][1]))
    for x, y in pts[1:]:
        p.append(inkex.paths.Line(x, y))
    if closed:
        p.append(inkex.paths.ZoneClose())
    return p


def set_style(el, **kwargs):
    st = el.style
    for k, v in kwargs.items():
        st[k] = v
    el.style = st


def path_to_csp_in_doc_coords(path_el: PathElement):
    tr = path_el.composed_transform()
    p = path_el.path.transform(tr)
    return CubicSuperPath(p)


def bounds_of_points(pts):
    xs = [p[0] for p in pts]
    ys = [p[1] for p in pts]
    return (min(xs), min(ys), max(xs), max(ys))


def iter_paths_under(node):
    for el in node.iter():
        if isinstance(el, PathElement):
            yield el


def path_is_closed(path_el: PathElement) -> bool:
    d = path_el.get("d")
    if not d:
        return False
    try:
        p = Path(d)
        for cmd in p:
            if cmd.letter in ("Z", "z"):
                return True
    except Exception:
        return ("Z" in d) or ("z" in d)
    return False


# ---------------------------
# Gradients (global userSpaceOnUse)
# ---------------------------
def ensure_defs(svg):
    if svg.defs is not None:
        return svg.defs
    defs = etree.SubElement(svg.getroot(), inkex.addNS("defs", "svg"))
    return defs


def add_linear_gradient(defs, gid, x1, y1, x2, y2, c1, c2):
    g = etree.SubElement(defs, inkex.addNS("linearGradient", "svg"))
    g.set("id", gid)
    g.set("gradientUnits", "userSpaceOnUse")
    g.set("x1", str(x1))
    g.set("y1", str(y1))
    g.set("x2", str(x2))
    g.set("y2", str(y2))

    s1 = etree.SubElement(g, inkex.addNS("stop", "svg"))
    s1.set("offset", "0%")
    s1.set("stop-color", rgb_to_css(c1))

    s2 = etree.SubElement(g, inkex.addNS("stop", "svg"))
    s2.set("offset", "100%")
    s2.set("stop-color", rgb_to_css(c2))
    return gid


def add_radial_gradient(defs, gid, cx, cy, r, c1, c2):
    g = etree.SubElement(defs, inkex.addNS("radialGradient", "svg"))
    g.set("id", gid)
    g.set("gradientUnits", "userSpaceOnUse")
    g.set("cx", str(cx))
    g.set("cy", str(cy))
    g.set("r", str(r))

    s1 = etree.SubElement(g, inkex.addNS("stop", "svg"))
    s1.set("offset", "0%")
    s1.set("stop-color", rgb_to_css(c1))

    s2 = etree.SubElement(g, inkex.addNS("stop", "svg"))
    s2.set("offset", "100%")
    s2.set("stop-color", rgb_to_css(c2))
    return gid


def find_first_paint_rgb(node):
    for p in iter_paths_under(node):
        st = p.style.get("stroke")
        rgb = parse_css_color_to_rgb(st)
        if rgb:
            return rgb
        fl = p.style.get("fill")
        rgb = parse_css_color_to_rgb(fl)
        if rgb:
            return rgb
    return None


def build_gradients(svg, L, T, R, B, ref_rgb, hue_delta_percent=13.0):
    """
    Crée (ou recrée) les gradients (bg radial + fill linear) et renvoie leurs ids.
    """
    if is_neutral_gray(ref_rgb):
        ref_rgb = (40, 140, 255)

    hue_delta_deg = 360.0 * (hue_delta_percent / 100.0)
    c_minus = rotate_hue_rgb(ref_rgb, -hue_delta_deg)
    c_plus  = rotate_hue_rgb(ref_rgb, +hue_delta_deg)

    opp = rotate_hue_rgb(ref_rgb, 180.0)
    bg_minus = rotate_hue_rgb(opp, -hue_delta_deg)
    bg_plus  = rotate_hue_rgb(opp, +hue_delta_deg)

    defs = ensure_defs(svg)

    gid_fill  = "__AUTO_LINEAR_FILL__"
    gid_bg    = "__AUTO_RADIAL_BG__"

    for child in list(defs):
        if child.get("id") in (gid_fill, gid_bg):
            defs.remove(child)

    cx = (L + R) / 2.0
    cy = (T + B) / 2.0
    diag_r = 0.6 * math.hypot(R - L, B - T)

    add_linear_gradient(defs, gid_fill, L, T, R, B, c_minus, c_plus)
    add_radial_gradient(defs, gid_bg, cx, cy, diag_r, bg_minus, bg_plus)

    return gid_fill, gid_bg


def apply_background(layer, L, T, R, B, gid_bg):
    rect = Rectangle()
    rect.set("x", str(L))
    rect.set("y", str(T))
    rect.set("width", str(R - L))
    rect.set("height", str(B - T))
    rect.style["fill"] = f"url(#{gid_bg})"
    rect.style["stroke"] = "none"
    rect.set(inkex.addNS("label", "inkscape"), "__AUTO_BACKGROUND__")
    layer.insert(0, rect)


def apply_fill_gradient_to_model(model, gid_fill):
    """
    ✅ Le correctif demandé :
    - applique le gradient en FILL dans le modèle (AVANT duplication)
    - enlève le contour si la forme est fermée
    """
    for p in iter_paths_under(model):
        if not p.get("d"):
            continue
        if path_is_closed(p):
            p.style["fill"] = f"url(#{gid_fill})"
            p.style["stroke"] = "none"
        else:
            # path ouvert -> on ne force pas de fill (sinon rendu bizarre)
            # (tu peux choisir de garder stroke none, mais par défaut on garde le stroke existant)
            if not p.style.get("fill"):
                p.style["fill"] = "none"


# ---------------------------
# Main Extension (SELECTION)
# ---------------------------
class TessalionSimplifyDuplicate(inkex.EffectExtension):
    def add_arguments(self, pars):
        pars.add_argument("--mode", default="A")

        # A
        pars.add_argument("--sample_a", type=float, default=3.0)
        pars.add_argument("--eps_a", type=float, default=6.0)
        pars.add_argument("--mindist_a", type=float, default=6.0)
        pars.add_argument("--force_closed_a", type=inkex.Boolean, default=True)

        # B
        pars.add_argument("--sample_b", type=float, default=3.0)
        pars.add_argument("--mindist_b", type=float, default=0.0)
        pars.add_argument("--crit_b", type=float, default=12.0)
        pars.add_argument("--margin_b", type=float, default=40.0)
        pars.add_argument("--treat_closed_b", type=inkex.Boolean, default=True)
        pars.add_argument("--op_min", type=float, default=8.0)
        pars.add_argument("--op_max", type=float, default=14.0)

    def effect(self):
        mode = (self.options.mode or "A").strip().upper()
        if mode == "A":
            self.run_mode_a_from_selection()
        else:
            self.run_mode_b_from_selection()

    def run_mode_a_from_selection(self):
        sel = list(self.svg.selection.values())
        paths = [e for e in sel if isinstance(e, PathElement)]
        if not paths:
            raise inkex.AbortExtension("Mode A : sélectionne au moins un chemin (Path).")

        step_px = max(0.2, float(self.options.sample_a))
        eps = max(0.1, float(self.options.eps_a))
        mind = max(0.0, float(self.options.mindist_a))
        force_closed = bool(self.options.force_closed_a)

        pts_all = []
        for pe in paths:
            csp = path_to_csp_in_doc_coords(pe)
            pts = sample_csp(csp, step_px, treat_closed=True)
            for p in pts:
                if not pts_all or dist(p, pts_all[-1]) > 1e-6:
                    pts_all.append(p)

        if len(pts_all) < 3:
            raise inkex.AbortExtension("Mode A : extraction points insuffisante (<3).")

        if mind > 0:
            pts_all = enforce_min_distance_sequential(pts_all, mind)

        pts_all = rdp(pts_all, eps)

        if force_closed and pts_all and not same_pt(pts_all[0], pts_all[-1]):
            pts_all.append(pts_all[0])

        out = PathElement()
        out.path = make_polyline_path(pts_all, closed=force_closed)
        set_style(out, fill="none", stroke="#000000", **{"stroke-width": "1"})
        self.svg.get_current_layer().add(out)

    def run_mode_b_from_selection(self):
        sel = list(self.svg.selection.values())
        if len(sel) < 2:
            raise inkex.AbortExtension(
                "Mode B : sélectionne au moins 2 éléments :\n"
                "1) le modèle à dupliquer (en premier)\n"
                "2) le support (en second) (path ou groupe contenant des paths)"
            )

        # ✅ modèle = 1er sélectionné
        model = sel[0]

        # ✅ support = 2e sélectionné uniquement (ça te rend le contrôle)
        support_node = sel[1]

        support_paths = []
        if isinstance(support_node, PathElement):
            support_paths.append(support_node)
        else:
            support_paths.extend(list(iter_paths_under(support_node)))

        if not support_paths:
            raise inkex.AbortExtension(
                "Mode B : le support (2e sélection) doit contenir au moins un Path.\n"
                "Astuce : groupe ton SVG support si besoin, puis sélectionne-le en 2e."
            )

        step_px = max(0.2, float(self.options.sample_b))
        mind = max(0.0, float(self.options.mindist_b))
        crit = max(0.0, float(self.options.crit_b))
        margin = max(0.0, float(self.options.margin_b))
        treat_closed = bool(self.options.treat_closed_b)

        op_min = clamp(float(self.options.op_min), 0, 100)
        op_max = clamp(float(self.options.op_max), 0, 100)
        if op_max < op_min:
            op_min, op_max = op_max, op_min

        raw_pts = []
        for pe in support_paths:
            csp = path_to_csp_in_doc_coords(pe)
            pts = sample_csp(csp, step_px, treat_closed=treat_closed)
            for p in pts:
                if not raw_pts or dist(p, raw_pts[-1]) > 1e-6:
                    raw_pts.append(p)

        if not raw_pts:
            raise inkex.AbortExtension("Mode B : aucun point extrait du support.")

        pts = raw_pts
        if mind > 0:
            pts = enforce_min_distance_sequential(pts, mind)
        if crit > 0:
            pts = filter_by_critical_radius_sequential(pts, crit)
        if not pts:
            raise inkex.AbortExtension("Mode B : après filtrage, aucun point restant.")

        # viewBox from points + margin
        L, T, R, B = bounds_of_points(pts)
        L -= margin
        T -= margin
        R += margin
        B += margin
        self.svg.set("viewBox", f"{L} {T} {R-L} {B-T}")

        layer = self.svg.get_current_layer()

        # ✅ crée gradients (fill + bg) à partir d'une couleur de référence trouvée dans le modèle
        ref = find_first_paint_rgb(model) or (0, 0, 0)
        gid_fill, gid_bg = build_gradients(self.svg, L, T, R, B, ref, hue_delta_percent=13.0)

        # ✅ background
        apply_background(layer, L, T, R, B, gid_bg)

        # ✅ applique le gradient de remplissage AU MODÈLE (avant duplication)
        apply_fill_gradient_to_model(model, gid_fill)

        # bbox modèle (après stylage)
        bb = model.bounding_box()
        if bb is None:
            raise inkex.AbortExtension("Mode B : bbox modèle impossible (groupe-le si besoin).")

        mCx = (bb.left + bb.right) / 2.0
        mCy = (bb.top + bb.bottom) / 2.0

        final_group = Group()
        final_group.set(inkex.addNS("label", "inkscape"), "__FINAL_DUPLICATES__")
        layer.add(final_group)

        for i, (x, y) in enumerate(pts):
            dup = model.copy()
            g = Group()
            g.set(inkex.addNS("label", "inkscape"), f"__DUP__{i}")
            g.add(dup)

            dx = x - mCx
            dy = y - mCy
            cur = dup.transform if dup.transform is not None else Transform()
            dup.transform = Transform(f"translate({dx},{dy})") @ cur

            op = random.uniform(op_min, op_max) / 100.0
            g.style["opacity"] = f"{op:.3f}"
            final_group.add(g)


if __name__ == "__main__":
    TessalionSimplifyDuplicate().run()
