#target illustrator

(function () {
  if (app.documents.length === 0) app.documents.add(DocumentColorSpace.RGB);

  // =========================================================
  // UI
  // =========================================================
  var w = new Window("dialog", "Simplifier + Dupliquer sur points (RDP + Rayon critique)");
  w.orientation = "column";
  w.alignChildren = "fill";

  var wrap = w.add("group");
  wrap.orientation = "row";
  wrap.alignChildren = "fill";

  // ----------------------------
  // A) Simplifier
  // ----------------------------
  var simpPanel = wrap.add("panel", undefined, "A) Simplifier un SVG (optionnel)");
  simpPanel.orientation = "column";
  simpPanel.alignChildren = "fill";

  var simpRow1 = simpPanel.add("group");
  simpRow1.orientation = "row";
  simpRow1.alignChildren = ["fill", "center"];
  simpRow1.add("statictext", undefined, "SVG à simplifier :");
  var simpTxt = simpRow1.add("edittext", undefined, "", { readonly: true });
  simpTxt.characters = 35;
  var simpChooseBtn = simpRow1.add("button", undefined, "Choisir…");

  var simpOpts = simpPanel.add("group");
  simpOpts.orientation = "column";
  simpOpts.alignChildren = "left";

  var g1 = simpOpts.add("group");
  g1.add("statictext", undefined, "Pas échantillonnage (px) :");
  var sampleTxtA = g1.add("edittext", undefined, "3.0"); sampleTxtA.characters = 7;

  var g2 = simpOpts.add("group");
  g2.add("statictext", undefined, "Tolérance RDP (px) :");
  var epsTxtA = g2.add("edittext", undefined, "6.0"); epsTxtA.characters = 7;

  var g3 = simpOpts.add("group");
  g3.add("statictext", undefined, "Distance min (px) :");
  var minDistTxtA = g3.add("edittext", undefined, "6.0"); minDistTxtA.characters = 7;

  var g4 = simpOpts.add("group");
  var closeChkA = g4.add("checkbox", undefined, "Forcer tracé fermé");
  closeChkA.value = true;

  var g5 = simpOpts.add("group");
  var outlineChkA = g5.add("checkbox", undefined, "Outline Stroke avant extraction");
  outlineChkA.value = true;

  var g6 = simpOpts.add("group");
  var uniteChkA = g6.add("checkbox", undefined, "Unifier (Pathfinder Unite) avant extraction");
  uniteChkA.value = true;

  var simpBtns = simpPanel.add("group");
  simpBtns.alignment = "right";
  var simpRunBtn = simpBtns.add("button", undefined, "Simplifier + Enregistrer…");

  // Séparateur
  var sep = wrap.add("panel");
  sep.preferredSize.width = 2;
  sep.minimumSize.width = 2;

  // ----------------------------
  // B) Duplication
  // ----------------------------
  var dupPanel = wrap.add("panel", undefined, "B) Dupliquer un SVG sur les points d’un SVG support");
  dupPanel.orientation = "column";
  dupPanel.alignChildren = "fill";

  var p1 = dupPanel.add("panel", undefined, "1) SVG support (extraction de points)");
  p1.orientation = "row";
  p1.alignChildren = ["fill","center"];
  var srcTxt = p1.add("edittext", undefined, "", { readonly:true }); srcTxt.characters = 35;
  var srcBtn = p1.add("button", undefined, "Choisir…");

  var p2 = dupPanel.add("panel", undefined, "2) SVG à dupliquer");
  p2.orientation = "row";
  p2.alignChildren = ["fill","center"];
  var dupTxt = p2.add("edittext", undefined, "", { readonly:true }); dupTxt.characters = 35;
  var dupBtn = p2.add("button", undefined, "Choisir…");

  var p3 = dupPanel.add("panel", undefined, "Options extraction + filtre points");
  p3.orientation = "column";
  p3.alignChildren = "left";

  var m1 = p3.add("group");
  m1.add("statictext", undefined, "Pas échantillonnage support (px) :");
  var sampleTxtB = m1.add("edittext", undefined, "3.0"); sampleTxtB.characters = 7;

  var m2 = p3.add("group");
  m2.add("statictext", undefined, "Distance min (px) :");
  var minDistTxtB = m2.add("edittext", undefined, "0"); minDistTxtB.characters = 7;

  var m3 = p3.add("group");
  m3.add("statictext", undefined, "Rayon critique (px) :");
  var critTxtB = m3.add("edittext", undefined, "12.0"); critTxtB.characters = 7;

  var m4 = p3.add("group");
  m4.add("statictext", undefined, "Marge plan de travail (px) :");
  var marginTxt = m4.add("edittext", undefined, "40"); marginTxt.characters = 7;

  var b1 = p3.add("group");
  var outlineChkB = b1.add("checkbox", undefined, "Outline Stroke support (recommandé)");
  outlineChkB.value = true;

  var b2 = p3.add("group");
  var uniteChkB = b2.add("checkbox", undefined, "Unifier support (Pathfinder Unite)");
  uniteChkB.value = true;

  var b3 = p3.add("group");
  var closeChkB = b3.add("checkbox", undefined, "Traiter support comme fermé");
  closeChkB.value = true;

  // ✅ Opacité “statistique” des duplicatas (8..14)
  var p4 = dupPanel.add("panel", undefined, "Opacité des contours dupliqués");
  p4.orientation = "row";
  p4.alignChildren = ["left","center"];
  p4.add("statictext", undefined, "Min (%) :");
  var opMinTxt = p4.add("edittext", undefined, "8"); opMinTxt.characters = 5;
  p4.add("statictext", undefined, "Max (%) :");
  var opMaxTxt = p4.add("edittext", undefined, "14"); opMaxTxt.characters = 5;

  var btns = w.add("group");
  btns.alignment = "right";
  var ok = btns.add("button", undefined, "Exécuter B");
  var cancel = btns.add("button", undefined, "Annuler");

  // Files
  var simpFile = null, srcFile = null, dupFile = null;

  // UI events
  simpChooseBtn.onClick = function () {
    simpFile = File.openDialog("Choisir un SVG à simplifier", "*.svg");
    if (simpFile) simpTxt.text = simpFile.fsName;
  };

  simpRunBtn.onClick = function () {
    if (!simpFile) { alert("Choisis d'abord un SVG à simplifier."); return; }

    var samplePx = parseFloat(sampleTxtA.text); if (isNaN(samplePx) || samplePx < 0.2) samplePx = 3.0;
    var eps = parseFloat(epsTxtA.text); if (isNaN(eps) || eps <= 0) eps = 6.0;
    var minDist = parseFloat(minDistTxtA.text); if (isNaN(minDist) || minDist < 0) minDist = 0;

    try {
      simplifySvgGlobalRDP(simpFile, samplePx, eps, minDist, closeChkA.value, outlineChkA.value, uniteChkA.value);
    } catch (e) {
      alert("Erreur simplification:\n" + e + "\n\n" + ($.stack || ""));
    }
  };

  srcBtn.onClick = function () {
    srcFile = File.openDialog("Choisir le SVG support (points)", "*.svg");
    if (srcFile) srcTxt.text = srcFile.fsName;
  };

  dupBtn.onClick = function () {
    dupFile = File.openDialog("Choisir le SVG à dupliquer", "*.svg");
    if (dupFile) dupTxt.text = dupFile.fsName;
  };

  cancel.onClick = function () { w.close(); };

  ok.onClick = function () {
    if (!srcFile || !dupFile) { alert("Tu dois choisir les deux SVG (support + à dupliquer)."); return; }

    var samplePx = parseFloat(sampleTxtB.text); if (isNaN(samplePx) || samplePx < 0.2) samplePx = 3.0;
    var minDist = parseFloat(minDistTxtB.text); if (isNaN(minDist) || minDist < 0) minDist = 0;
    var crit = parseFloat(critTxtB.text); if (isNaN(crit) || crit < 0) crit = 0;
    var margin = parseFloat(marginTxt.text); if (isNaN(margin)) margin = 40;

    var opMin = parseFloat(opMinTxt.text); if (isNaN(opMin)) opMin = 8;
    var opMax = parseFloat(opMaxTxt.text); if (isNaN(opMax)) opMax = 14;
    opMin = clamp(opMin, 0, 100);
    opMax = clamp(opMax, 0, 100);
    if (opMax < opMin) { var tmp = opMin; opMin = opMax; opMax = tmp; }

    w.close();
    try { app.executeMenuCommand("doc-color-rgb"); } catch(_) {}

    try {
      duplicateOnSupportPoints(srcFile, dupFile, margin, samplePx, minDist, crit, closeChkB.value, outlineChkB.value, uniteChkB.value, opMin, opMax);
      alert("Terminé.");
    } catch (e) {
      alert("Erreur: " + e + "\nFichier: " + (e.fileName || "?") + "\nLigne: " + (e.line || "?") + "\n\n" + ($.stack || "?"));
    }
  };

  w.center();
  w.show();

  // =========================================================
  // A) Simplify (RDP global)
  // =========================================================
  function simplifySvgGlobalRDP(svgFile, samplePx, eps, minDist, forceClosed, doOutline, doUnite) {
    var doc = app.open(svgFile);
    app.activeDocument = doc;

    try { app.executeMenuCommand("expandStyle"); } catch(_) {}
    try { app.executeMenuCommand("expand"); } catch(_) {}

    if (doOutline) {
      try { selectAllUsablePaths(doc); app.executeMenuCommand("outline"); } catch(_) {}
      try { app.executeMenuCommand("expandStyle"); } catch(_) {}
      try { app.executeMenuCommand("expand"); } catch(_) {}
    }

    if (doUnite) {
      try {
        selectAllUsablePaths(doc);
        try { app.executeMenuCommand("group"); } catch(_){}
        app.executeMenuCommand("Live Pathfinder Add");
        try { app.executeMenuCommand("expandStyle"); } catch(_) {}
        try { app.executeMenuCommand("expand"); } catch(_) {}
        try { app.executeMenuCommand("ungroup"); } catch(_) {}
      } catch(_) {}
    }

    app.redraw();

    var pts = extractAllSampledPoints(doc, samplePx);
    if (pts.length < 3) { doc.close(SaveOptions.DONOTSAVECHANGES); throw new Error("Extraction points insuffisante."); }

    if (minDist > 0) pts = enforceMinDistanceSequential(pts, minDist);

    var beforeN = pts.length;
    pts = rdp(pts, eps);
    var afterN = pts.length;

    var layer = doc.activeLayer;
    try { clearLayer(layer); } catch(_) {}

    var p = layer.pathItems.add();
    p.stroked = true; p.filled = false; p.strokeWidth = 1;

    if (forceClosed) {
      if (!samePt(pts[0], pts[pts.length - 1])) pts.push([pts[0][0], pts[0][1]]);
      p.closed = true;
    } else p.closed = false;

    p.setEntirePath(pts);
    for (var i=0; i<p.pathPoints.length; i++) {
      p.pathPoints[i].pointType = PointType.CORNER;
      p.pathPoints[i].leftDirection  = p.pathPoints[i].anchor;
      p.pathPoints[i].rightDirection = p.pathPoints[i].anchor;
    }

    app.redraw();

    var outFile = File.saveDialog("Enregistrer une COPIE simplifiée (SVG)\nPoints: " + beforeN + " → " + afterN, "*.svg");
    if (!outFile) { doc.close(SaveOptions.DONOTSAVECHANGES); return; }

    var opt = new ExportOptionsSVG();
    opt.embedRasterImages = true;
    doc.exportFile(outFile, ExportType.SVG, opt);
    doc.close(SaveOptions.DONOTSAVECHANGES);

    alert("Copie simplifiée enregistrée :\n" + outFile.fsName);
  }

  // =========================================================
  // B) Duplicate on support points + critical radius
  // ✅ NO TAG : support imported in a TEMP LAYER, then layer removed
  // =========================================================
  function duplicateOnSupportPoints(srcSvg, dupSvg, margin, samplePx, minDist, critRadius, treatClosed, doOutline, doUnite, opMin, opMax) {
    var doc = app.activeDocument;

    // ✅ Calque final = calque actif au moment du lancement
    var finalLayer = doc.activeLayer;

    // ✅ Calque temporaire qui contiendra TOUT le support (même après Outline/Unite)
    var supportLayer = doc.layers.add();
    supportLayer.name = "__TMP_SUPPORT__";

    // ---- 1) Import support dans le calque temporaire
    doc.activeLayer = supportLayer;
    var srcGroup = importSVGIntoDoc(doc, srcSvg);
    srcGroup.name = "__SRC_SUPPORT__";

    centerItemOnActiveArtboard(doc, srcGroup);
    fitActiveArtboardToItem(doc, srcGroup, margin);

    // Expand / Outline / Unite sur le support (dans supportLayer)
    doc.selection = null;
    srcGroup.selected = true;
    try { app.executeMenuCommand("expandStyle"); } catch(_) {}
    try { app.executeMenuCommand("expand"); } catch(_) {}

    if (doOutline) {
      try { doc.selection = null; srcGroup.selected = true; app.executeMenuCommand("outline"); } catch(_) {}
      try { app.executeMenuCommand("expandStyle"); } catch(_) {}
      try { app.executeMenuCommand("expand"); } catch(_) {}
    }
    if (doUnite) {
      try {
        doc.selection = null;
        srcGroup.selected = true;
        try { app.executeMenuCommand("group"); } catch(_){}
        app.executeMenuCommand("Live Pathfinder Add");
        try { app.executeMenuCommand("expandStyle"); } catch(_) {}
        try { app.executeMenuCommand("expand"); } catch(_) {}
        try { app.executeMenuCommand("ungroup"); } catch(_) {}
      } catch(_) {}
    }
    doc.selection = null;

    // ---- 2) Extract points
    var rawPts = extractSampledPointsFromGroup(srcGroup, samplePx, treatClosed);
    if (rawPts.length === 0) throw new Error("Aucun point trouvé dans le support.");

    var pts = rawPts;
    if (minDist > 0) pts = enforceMinDistanceSequential(pts, minDist);
    if (critRadius > 0) pts = filterByCriticalRadiusSequential(pts, critRadius);
    if (pts.length === 0) throw new Error("Après filtrage (minDist/rayon critique), aucun point restant.");

    // ---- 3) Import modèle + duplications dans le calque final
    doc.activeLayer = finalLayer;

    var masterGroup = importSVGIntoDoc(doc, dupSvg);
    masterGroup.name = "__MASTER_DUP__";

    var ab = activeArtboardRect(doc);
    masterGroup.left = ab[0] - 10000;
    masterGroup.top  = ab[1] + 10000;

    var finalGroup = doc.activeLayer.groupItems.add();
    finalGroup.name = "__FINAL_DUPLICATES__";

    var duplicates = [];
    for (var i = 0; i < pts.length; i++) {
      var d = masterGroup.duplicate(finalGroup, ElementPlacement.PLACEATEND);
      d.name = "__DUP__" + i;
      moveItemCenterToPoint(d, pts[i][0], pts[i][1]);
      duplicates.push(d);
    }

    // cleanup master
    try { masterGroup.remove(); } catch(_) {}

    // ---- 4) Styles
    applyGlobalGradients(doc, finalGroup, 13);
    applyRandomOpacity(finalGroup, opMin, opMax);

    // ✅ Fit duplicates inside artboard (without distortion)
    fitGroupToActiveArtboard(doc, finalGroup, margin);

    // ---- ✅ 5) Suppression DEFINITIVE du support : on supprime le calque
    try { supportLayer.remove(); } catch(_) {}

    // ---- 6) Select duplicates
    doc.selection = null;
    for (var k = 0; k < duplicates.length; k++) duplicates[k].selected = true;
  }

  // =========================================================
  // ✅ Random opacity only (8..14)
  // =========================================================
  function applyRandomOpacity(container, minPct, maxPct) {
    var items = collectPageItems(container);
    for (var i=0; i<items.length; i++) {
      var it = items[i];
      if (!it || it.locked || it.hidden) continue;
      if (it.guides || it.clipping) continue;

      if (it.typename === "PathItem") {
        try { it.opacity = rand(minPct, maxPct); } catch(_) {}
      } else if (it.typename === "CompoundPathItem") {
        try { it.opacity = rand(minPct, maxPct); } catch(_) {}
        try {
          for (var k=0; k<it.pathItems.length; k++) {
            try { it.pathItems[k].opacity = rand(minPct, maxPct); } catch(_) {}
          }
        } catch(_) {}
      } else if (it.typename === "GroupItem") {
        try { it.opacity = rand(minPct, maxPct); } catch(_) {}
      }
    }
  }

  function rand(a,b){ return a + Math.random() * (b-a); }
  function clamp(v, lo, hi){ return Math.max(lo, Math.min(hi, v)); }

  // =========================================================
  // Support extraction helpers (sampling inside group)
  // =========================================================
  function extractSampledPointsFromGroup(groupItem, stepPx, treatClosed) {
    var out = [];
    var paths = groupItem.pathItems;
    for (var i=0; i<paths.length; i++) {
      var p = paths[i];
      if (!p || p.locked || p.hidden) continue;
      if (p.guides || p.clipping) continue;

      var pts = samplePathItem(p, stepPx, treatClosed ? true : p.closed);
      for (var k=0; k<pts.length; k++) {
        if (out.length === 0 || dist(out[out.length-1], pts[k]) > 1e-6) out.push(pts[k]);
      }
    }
    if (groupItem.groupItems && groupItem.groupItems.length) {
      for (var g=0; g<groupItem.groupItems.length; g++) {
        var sub = extractSampledPointsFromGroup(groupItem.groupItems[g], stepPx, treatClosed);
        for (var s=0; s<sub.length; s++) {
          if (out.length === 0 || dist(out[out.length-1], sub[s]) > 1e-6) out.push(sub[s]);
        }
      }
    }
    return out;
  }

  // =========================================================
  // Global extraction (A)
  // =========================================================
  function extractAllSampledPoints(doc, stepPx) {
    var out = [];
    for (var i=0; i<doc.pathItems.length; i++) {
      var path = doc.pathItems[i];
      if (!path || path.locked || path.hidden) continue;
      if (path.guides || path.clipping) continue;

      var pts = samplePathItem(path, stepPx, path.closed);
      for (var k=0; k<pts.length; k++) {
        if (out.length === 0 || dist(out[out.length-1], pts[k]) > 1e-6) out.push(pts[k]);
      }
    }
    return out;
  }

  // =========================================================
  // Sampling (shared)
  // =========================================================
  function samplePathItem(pathItem, stepPx, closedOverride) {
    var out = [];
    var pts = pathItem.pathPoints;
    if (!pts || pts.length === 0) return out;

    var closed = !!closedOverride;
    var n = pts.length;

    for (var i=0; i<n; i++) {
      if (!closed && i === n-1) break;

      var a = pts[i];
      var b = pts[(i+1) % n];

      var P0 = a.anchor;
      var P1 = a.rightDirection;
      var P2 = b.leftDirection;
      var P3 = b.anchor;

      if (out.length === 0) out.push([P0[0], P0[1]]);

      var isCurve = !(samePt2(P0, P1) && samePt2(P2, P3));
      if (!isCurve) {
        out.push([P3[0], P3[1]]);
      } else {
        var approxLen = approxBezierLen(P0,P1,P2,P3);
        var steps = Math.max(2, Math.ceil(approxLen / stepPx));
        for (var s=1; s<=steps; s++) {
          var t = s/steps;
          out.push(bezierPoint(P0,P1,P2,P3,t));
        }
      }
    }
    return out;
  }

  function approxBezierLen(P0,P1,P2,P3){
    var prev = [P0[0],P0[1]];
    var len = 0;
    for (var i=1; i<=12; i++){
      var t = i/12;
      var p = bezierPoint(P0,P1,P2,P3,t);
      len += dist(prev, p);
      prev = p;
    }
    return len;
  }

  function bezierPoint(P0,P1,P2,P3,t){
    var mt = 1-t;
    var a = mt*mt*mt;
    var b = 3*mt*mt*t;
    var c = 3*mt*t*t;
    var d = t*t*t;
    return [
      a*P0[0] + b*P1[0] + c*P2[0] + d*P3[0],
      a*P0[1] + b*P1[1] + c*P2[1] + d*P3[1]
    ];
  }

  // =========================================================
  // Filters
  // =========================================================
  function enforceMinDistanceSequential(pts, minDist) {
    if (!pts || pts.length === 0) return pts;
    var out = [pts[0]];
    for (var i=1; i<pts.length; i++) {
      if (dist(pts[i], out[out.length-1]) >= minDist) out.push(pts[i]);
    }
    return out;
  }

  function filterByCriticalRadiusSequential(pts, critRadius) {
    if (!pts || pts.length === 0) return pts;
    var out = [];
    var i = 0;

    while (i < pts.length) {
      var ref = pts[i];
      out.push(ref);
      i++;
      while (i < pts.length && dist(pts[i], ref) < critRadius) i++;
    }
    return out;
  }

  // =========================================================
  // RDP (A)
  // =========================================================
  function rdp(points, epsilon) {
    if (!points || points.length < 3) return points;
    var closed = samePt(points[0], points[points.length-1]);
    var work = closed ? points.slice(0, points.length-1) : points;

    var keep = new Array(work.length);
    for (var i=0;i<keep.length;i++) keep[i] = false;
    keep[0] = true; keep[work.length-1] = true;

    rdpRec(work, 0, work.length-1, epsilon*epsilon, keep);

    var out = [];
    for (var k=0;k<work.length;k++) if (keep[k]) out.push(work[k]);
    if (closed) out.push([out[0][0], out[0][1]]);
    return out;
  }

  function rdpRec(pts, a, b, epsSq, keep) {
    if (b <= a+1) return;

    var maxD = -1, idx = -1;
    var A = pts[a], B = pts[b];

    for (var i=a+1; i<b; i++) {
      var d = pointLineDistSq(pts[i], A, B);
      if (d > maxD) { maxD = d; idx = i; }
    }

    if (maxD > epsSq && idx !== -1) {
      keep[idx] = true;
      rdpRec(pts, a, idx, epsSq, keep);
      rdpRec(pts, idx, b, epsSq, keep);
    }
  }

  function pointLineDistSq(P, A, B) {
    var vx = B[0]-A[0], vy = B[1]-A[1];
    var wx = P[0]-A[0], wy = P[1]-A[1];

    var c1 = vx*wx + vy*wy;
    if (c1 <= 0) return dist2(P, A);

    var c2 = vx*vx + vy*vy;
    if (c2 <= c1) return dist2(P, B);

    var t = c1 / c2;
    var proj = [A[0] + t*vx, A[1] + t*vy];
    return dist2(P, proj);
  }

  // =========================================================
  // Import SVG into active doc (+ expand)
  // =========================================================
  function importSVGIntoDoc(targetDoc, svgFile) {
    var tmpDoc = app.open(svgFile);

    app.activeDocument = targetDoc;
    var container = targetDoc.activeLayer.groupItems.add();
    container.name = "__IMPORTED_SVG__";

    var okCount = 0;
    for (var i = tmpDoc.pageItems.length - 1; i >= 0; i--) {
      var it = tmpDoc.pageItems[i];
      try {
        if (!it || it.locked || it.hidden) continue;
        app.activeDocument = tmpDoc;
        var dup = it.duplicate(targetDoc.activeLayer, ElementPlacement.PLACEATBEGINNING);
        app.activeDocument = targetDoc;
        dup.move(container, ElementPlacement.PLACEATEND);
        okCount++;
      } catch (e) {}
    }

    app.activeDocument = tmpDoc;
    tmpDoc.close(SaveOptions.DONOTSAVECHANGES);

    if (okCount === 0) throw new Error("Import SVG: aucun élément duplicable.");

    app.activeDocument = targetDoc;
    targetDoc.selection = null;
    container.selected = true;
    try { app.executeMenuCommand("expandStyle"); } catch(_) {}
    try { app.executeMenuCommand("expand"); } catch(_) {}
    targetDoc.selection = null;

    return container;
  }

  // =========================================================
  // Selection / cleanup helpers
  // =========================================================
  function selectAllUsablePaths(doc){
    doc.selection = null;
    for (var i=0;i<doc.pathItems.length;i++){
      var p = doc.pathItems[i];
      if (!p || p.locked || p.hidden) continue;
      if (p.guides || p.clipping) continue;
      try { p.selected = true; } catch(_){}
    }
  }

  function clearLayer(layer){
    for (var i=layer.pageItems.length-1; i>=0; i--) {
      try { layer.pageItems[i].remove(); } catch(_){}
    }
  }

  // =========================================================
  // Artboard helpers
  // =========================================================
  function activeArtboardRect(doc) {
    return doc.artboards[doc.artboards.getActiveArtboardIndex()].artboardRect; // [L,T,R,B]
  }

  function centerItemOnActiveArtboard(doc, item) {
    var ab = activeArtboardRect(doc);
    var abCx = (ab[0] + ab[2]) / 2;
    var abCy = (ab[1] + ab[3]) / 2;

    var b = item.visibleBounds;
    var itCx = (b[0] + b[2]) / 2;
    var itCy = (b[1] + b[3]) / 2;

    item.left += (abCx - itCx);
    item.top  += (abCy - itCy);
  }

  function fitActiveArtboardToItem(doc, item, margin) {
    var b = item.visibleBounds;
    var rect = [b[0]-margin, b[1]+margin, b[2]+margin, b[3]-margin];
    doc.artboards[doc.artboards.getActiveArtboardIndex()].artboardRect = rect;
  }

  function moveItemCenterToPoint(item, x, y) {
    var b = item.visibleBounds;
    var cx = (b[0] + b[2]) / 2;
    var cy = (b[1] + b[3]) / 2;
    item.left += (x - cx);
    item.top  += (y - cy);
  }

  // =========================================================
  // Math helpers
  // =========================================================
  function dist(a,b){ var dx=a[0]-b[0], dy=a[1]-b[1]; return Math.sqrt(dx*dx+dy*dy); }
  function dist2(a,b){ var dx=a[0]-b[0], dy=a[1]-b[1]; return dx*dx+dy*dy; }
  function samePt(a,b){ return Math.abs(a[0]-b[0])<1e-4 && Math.abs(a[1]-b[1])<1e-4; }
  function samePt2(A,B){ return Math.abs(A[0]-B[0])<1e-4 && Math.abs(A[1]-B[1])<1e-4; }

  // =========================================================
  // ✅ COLOR / GRADIENT GLOBAL APPLY (unchanged logic)
  // =========================================================
  function applyGlobalGradients(doc, rootItemOrNull, hueDeltaPercent) {
    var hueDeltaDeg = 360 * (hueDeltaPercent / 100.0);

    var refRGB = findFirstStrokeRGB(doc, rootItemOrNull) || [0, 0, 0];

    var cMinus = rotateHueRGB(refRGB, -hueDeltaDeg);
    var cPlus  = rotateHueRGB(refRGB,  hueDeltaDeg);

    var oppRGB = rotateHueRGB(refRGB, 180);
    var bgMinus = rotateHueRGB(oppRGB, -hueDeltaDeg);
    var bgPlus  = rotateHueRGB(oppRGB,  hueDeltaDeg);

    var gradLinear = getOrCreateGradient(doc, "__AUTO_LINEAR_STROKE__", false, cMinus, cPlus);
    var gradRadial = getOrCreateGradient(doc, "__AUTO_RADIAL_BG__", true, bgMinus, bgPlus);

    applyFillGradientAndRemoveStroke(rootItemOrNull || doc, gradLinear);
    ensureBackgroundRect(doc, gradRadial);
  }

  function applyFillGradientAndRemoveStroke(container, gradientObj) {
    var items = collectPageItems(container);
    for (var i = 0; i < items.length; i++) {
      var it = items[i];
      if (!it || it.locked || it.hidden) continue;
      if (it.guides || it.clipping) continue;

      if (it.typename === "PathItem") {
        if (!it.filled && !it.stroked) continue;

        it.filled = true;
        var fill = new GradientColor();
        fill.gradient = gradientObj;
        it.fillColor = fill;

        it.stroked = false;
      } else if (it.typename === "CompoundPathItem") {
        try {
          for (var k = 0; k < it.pathItems.length; k++) {
            var p = it.pathItems[k];
            if (!p) continue;
            p.filled = true;
            var fc = new GradientColor();
            fc.gradient = gradientObj;
            p.fillColor = fc;
            p.stroked = false;
          }
        } catch(_) {}
      }
    }
  }

  function ensureBackgroundRect(doc, gradientObj) {
    var layer = doc.activeLayer;
    var ab = doc.artboards[doc.artboards.getActiveArtboardIndex()].artboardRect;
    var L = ab[0], T = ab[1], R = ab[2], B = ab[3];
    var W = R - L, H = T - B;

    var rect = layer.pathItems.rectangle(T, L, W, H);
    rect.name = "__AUTO_BACKGROUND__";
    rect.stroked = false;
    rect.filled = true;

    var fill = new GradientColor();
    fill.gradient = gradientObj;
    rect.fillColor = fill;

    try { rect.zOrder(ZOrderMethod.SENDTOBACK); } catch(_) {}
  }

  function collectPageItems(container) {
    var out = [];
    try {
      if (container.pageItems && container.pageItems.length) {
        for (var i=0; i<container.pageItems.length; i++) out.push(container.pageItems[i]);
      }
    } catch(_) {}
    try {
      if (container.groupItems && container.groupItems.length) {
        for (var g=0; g<container.groupItems.length; g++) {
          var sub = collectPageItems(container.groupItems[g]);
          for (var k=0; k<sub.length; k++) out.push(sub[k]);
        }
      }
    } catch(_) {}
    try {
      if (container.compoundPathItems && container.compoundPathItems.length) {
        for (var c=0; c<container.compoundPathItems.length; c++) out.push(container.compoundPathItems[c]);
      }
    } catch(_) {}
    return out;
  }

  function findFirstStrokeRGB(doc, rootItemOrNull) {
    var items = collectPageItems(rootItemOrNull || doc);
    for (var i=0; i<items.length; i++) {
      var it = items[i];
      if (!it || it.locked || it.hidden) continue;
      if (it.guides || it.clipping) continue;

      if (it.typename === "PathItem") {
        if (it.stroked && it.strokeColor) {
          var rgb = toRGBArray(it.strokeColor);
          if (rgb) return rgb;
        }
      } else if (it.typename === "CompoundPathItem") {
        try {
          for (var k=0; k<it.pathItems.length; k++) {
            var p = it.pathItems[k];
            if (p && p.stroked && p.strokeColor) {
              var rgb2 = toRGBArray(p.strokeColor);
              if (rgb2) return rgb2;
            }
          }
        } catch(_) {}
      }
    }
    return null;
  }

  function getOrCreateGradient(doc, name, radial, rgbA, rgbB) {
    for (var i=0; i<doc.gradients.length; i++) {
      if (doc.gradients[i].name === name) {
        var g = doc.gradients[i];
        setupGradientStops(g, radial, rgbA, rgbB);
        return g;
      }
    }
    var grad = doc.gradients.add();
    grad.name = name;
    setupGradientStops(grad, radial, rgbA, rgbB);
    return grad;
  }

  function setupGradientStops(grad, radial, rgbA, rgbB) {
    grad.type = radial ? GradientType.RADIAL : GradientType.LINEAR;

    while (grad.gradientStops.length < 2) grad.gradientStops.add();
    while (grad.gradientStops.length > 2) grad.gradientStops[grad.gradientStops.length-1].remove();

    grad.gradientStops[0].rampPoint = 0;
    grad.gradientStops[1].rampPoint = 100;

    grad.gradientStops[0].color = makeRGBColor(rgbA);
    grad.gradientStops[1].color = makeRGBColor(rgbB);
  }

  function toRGBArray(colorObj) {
    if (!colorObj) return null;

    if (colorObj.typename === "RGBColor") {
      return [clamp255(colorObj.red), clamp255(colorObj.green), clamp255(colorObj.blue)];
    }
    if (colorObj.typename === "GrayColor") {
      var v = clamp255((1 - (colorObj.gray/100)) * 255);
      return [v, v, v];
    }
    if (colorObj.typename === "CMYKColor") {
      return cmykToRgb(colorObj.cyan, colorObj.magenta, colorObj.yellow, colorObj.black);
    }
    try {
      if (colorObj.spot && colorObj.spot.color) return toRGBArray(colorObj.spot.color);
    } catch(_) {}
    return null;
  }

  function makeRGBColor(rgb) {
    var c = new RGBColor();
    c.red = clamp255(rgb[0]);
    c.green = clamp255(rgb[1]);
    c.blue = clamp255(rgb[2]);
    return c;
  }

  function cmykToRgb(C,M,Y,K){
    var c = C/100, m = M/100, y = Y/100, k = K/100;
    var r = 255 * (1 - c) * (1 - k);
    var g = 255 * (1 - m) * (1 - k);
    var b = 255 * (1 - y) * (1 - k);
    return [clamp255(r), clamp255(g), clamp255(b)];
  }

  function clamp255(v){ return Math.max(0, Math.min(255, Math.round(v))); }

  function rotateHueRGB(rgb, deg){
    var hsl = rgbToHsl(rgb[0], rgb[1], rgb[2]);
    hsl[0] = (hsl[0] + deg) % 360;
    if (hsl[0] < 0) hsl[0] += 360;
    return hslToRgb(hsl[0], hsl[1], hsl[2]);
  }

  function rgbToHsl(r,g,b){
    r/=255; g/=255; b/=255;
    var max = Math.max(r,g,b), min = Math.min(r,g,b);
    var h, s, l = (max + min) / 2;

    if (max === min) {
      h = s = 0;
    } else {
      var d = max - min;
      s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
      switch(max){
        case r: h = (g - b) / d + (g < b ? 6 : 0); break;
        case g: h = (b - r) / d + 2; break;
        case b: h = (r - g) / d + 4; break;
      }
      h *= 60;
    }
    return [h, s, l];
  }

  function hslToRgb(h,s,l){
    h /= 360;
    var r,g,b;

    if (s === 0) {
      r = g = b = l;
    } else {
      var q = l < 0.5 ? l * (1 + s) : l + s - l*s;
      var p = 2*l - q;
      r = hue2rgb(p, q, h + 1/3);
      g = hue2rgb(p, q, h);
      b = hue2rgb(p, q, h - 1/3);
    }
    return [clamp255(r*255), clamp255(g*255), clamp255(b*255)];
  }

  function hue2rgb(p, q, t){
    if (t < 0) t += 1;
    if (t > 1) t -= 1;
    if (t < 1/6) return p + (q - p) * 6 * t;
    if (t < 1/2) return q;
    if (t < 2/3) return p + (q - p) * (2/3 - t) * 6;
    return p;
  }

  function fitGroupToActiveArtboard(doc, groupItem, insetPx) {
    insetPx = (isNaN(insetPx) ? 0 : insetPx);

    var ab = activeArtboardRect(doc); // [L,T,R,B]
    var L = ab[0] + insetPx;
    var T = ab[1] - insetPx;
    var R = ab[2] - insetPx;
    var B = ab[3] + insetPx;

    var abW = R - L;
    var abH = T - B;

    var b = groupItem.visibleBounds; // [left, top, right, bottom]
    var gW = b[2] - b[0];
    var gH = b[1] - b[3];

    if (gW <= 0 || gH <= 0) return;

    var s = Math.min(abW / gW, abH / gH);
    if (!isFinite(s) || s <= 0) return;

    // Ne jamais agrandir (optionnel). Si tu veux autoriser l’agrandissement, commente ce if.
    if (s > 1) s = 1;

    groupItem.resize(
      s * 100, s * 100,
      true,  // changePositions
      true,  // changeFillPatterns
      true,  // changeFillGradients
      true,  // changeStrokePatterns
      true,  // changeLineWidths
      Transformation.CENTER
    );

    centerItemOnActiveArtboard(doc, groupItem);
  }


})();
